<?php
/**
* @version	$Id: i_db_connection.php 15904 2013-07-17 18:52:50Z erik $
* @package	In-Portal
* @copyright	Copyright (C) 1997 - 2013 Intechnic. All rights reserved.
* @license      GNU/GPL
* In-Portal is Open Source software.
* This means that this software may have been modified pursuant
* the GNU General Public License, and as distributed it includes
* or is derivative of works licensed under the GNU General Public License
* or other free or open source software licenses.
* See http://www.in-portal.org/license for copyright notices and details.
*/

	defined('FULL_PATH') or die('restricted access!');

	/**
	 * Database connection interface
	 *
	 */
	interface IDBConnection {

		/**
		 * Checks if previous query execution raised an error.
		 *
		 * @return bool
		 * @access public
		 */
		public function hasError();

		/**
		 * Checks if connection to database is opened.
		 *
		 * @return bool
		 * @access public
		 */
		public function connectionOpened();

		/**
		 * Setups the connection according given configuration.
		 *
		 * @param Array $config
		 * @return bool
		 * @access public
		 */
		public function setup($config);

		/**
		 * Returns first field of first line of recordset if query ok or false otherwise.
		 *
		 * @param string $sql
		 * @param int $offset
		 * @return string
		 * @access public
		 */
		public function GetOne($sql, $offset = 0);

		/**
		 * Returns first row of recordset if query ok, false otherwise.
		 *
		 * @param string $sql
		 * @param int $offset
		 * @return Array
		 * @access public
		 */
		public function GetRow($sql, $offset = 0);

		/**
		 * Returns 1st column of recordset as one-dimensional array or false otherwise.
		 *
		 * Optional parameter $key_field can be used to set field name to be used as resulting array key.
		 *
		 * @param string $sql
		 * @param string $key_field
		 * @return Array
		 * @access public
		 */
		public function GetCol($sql, $key_field = null);

		/**
		 * Returns iterator for 1st column of a recordset or false in case of error.
		 *
		 * Optional parameter $key_field can be used to set field name to be used as resulting array key.
		 *
		 * @param string $sql
		 * @param string $key_field
		 * @return bool|kMySQLQueryCol
		 */
		public function GetColIterator($sql, $key_field = null);

		/**
		 * Queries db with $sql query supplied and returns rows selected if any, false otherwise.
		 *
		 * Optional parameter $key_field allows to set one of the query fields value as key in string array.
		 *
		 * @param string $sql
		 * @param string $key_field
		 * @param bool $no_debug
		 * @return Array
		 * @access public
		 */
		public function Query($sql, $key_field = null, $no_debug = false);

		/**
		 * Returns iterator to a recordset, produced from running $sql query.
		 *
		 * Queries db with $sql query supplied and returns kMySQLQuery iterator or false in case of error.
		 * Optional parameter $key_field allows to set one of the query fields value as key in string array.
		 *
		 * @param string $sql
		 * @param string $key_field
		 * @param bool   $no_debug
		 * @param string $iterator_class
		 * @return kMySQLQuery|bool
		 * @access public
		 */
		public function GetIterator($sql, $key_field = null, $no_debug = false, $iterator_class = 'kMySQLQuery');

		/**
		 * Free memory used to hold recordset handle.
		 *
		 * @access public
		 */
		public function Destroy();

		/**
		 * Performs sql query, that will change database content.
		 *
		 * @param string $sql
		 * @return bool
		 * @access public
		 */
		public function ChangeQuery($sql);

		/**
		 * Returns auto increment field value from insert like operation if any, zero otherwise.
		 *
		 * @return int
		 * @access public
		 */
		public function getInsertID();

		/**
		 * Returns row count affected by last query.
		 *
		 * @return int
		 * @access public
		 */
		public function getAffectedRows();

		/**
		 * Returns LIMIT sql clause part for specific db.
		 *
		 * @param int $offset
		 * @param int $rows
		 * @return string
		 * @access public
		 */
		public function getLimitClause($offset, $rows);

		/**
		 * If it's a string, adds quotes and backslashes. Otherwise returns as-is.
		 *
		 * @param mixed $string
		 * @return string
		 * @access public
		 */
		public function qstr($string);

		/**
		 * Calls "qstr" function for each given array element.
		 *
		 * @param Array $array
		 * @param string $function
		 * @return Array
		 */
		public function qstrArray($array, $function = 'qstr');

		/**
		 * Escapes string.
		 *
		 * @param mixed $string
		 * @return string
		 * @access public
		 */
		public function escape($string);

		/**
		 * Returns last error code occurred.
		 *
		 * @return int
		 * @access public
		 */
		public function getErrorCode();

		/**
		 * Returns last error message.
		 *
		 * @return string
		 * @access public
		 */
		public function getErrorMsg();

		/**
		 * Performs insert of given data (useful with small number of queries)
		 * or stores it to perform multiple insert later (useful with large number of queries).
		 *
		 * @param Array $fields_hash
		 * @param string $table
		 * @param string $type
		 * @param bool $insert_now
		 * @return bool
		 * @access public
		 */
		public function doInsert($fields_hash, $table, $type = 'INSERT', $insert_now = true);

		/**
		 * Update given field values to given record using $key_clause.
		 *
		 * @param Array $fields_hash
		 * @param string $table
		 * @param string $key_clause
		 * @return bool
		 * @access public
		 */
		public function doUpdate($fields_hash, $table, $key_clause);

		/**
		 * Allows to detect table's presence in database.
		 *
		 * @param string $table_name
		 * @param bool $force
		 * @return bool
		 * @access public
		 */
		public function TableFound($table_name, $force = false);

		/**
		 * Returns query processing statistics.
		 *
		 * @return Array
		 * @access public
		 */
		public function getQueryStatistics();

		/**
		 * Get status information from SHOW STATUS in an associative array.
		 *
		 * @param string $which
		 * @return Array
		 * @access public
		 */
		public function getStatus($which = '%');

		/**
		 * Get slave replication lag. It will only work if the DB user has the PROCESS privilege.
		 *
		 * @return int
		 * @access public
		 */
		public function getSlaveLag();
	}
